<?php

namespace Square\Types;

use Square\Core\Json\JsonSerializableType;
use Square\Core\Json\JsonProperty;
use Square\Core\Types\ArrayType;
use Square\Core\Types\Union;

/**
 * Contains details about how to fulfill this order.
 * Orders can only be created with at most one fulfillment using the API.
 * However, orders returned by the Orders API might contain multiple fulfillments because sellers can create multiple fulfillments using Square products such as Square Online.
 */
class Fulfillment extends JsonSerializableType
{
    /**
     * @var ?string $uid A unique ID that identifies the fulfillment only within this order.
     */
    #[JsonProperty('uid')]
    private ?string $uid;

    /**
     * The type of the fulfillment.
     * See [FulfillmentType](#type-fulfillmenttype) for possible values
     *
     * @var ?value-of<FulfillmentType> $type
     */
    #[JsonProperty('type')]
    private ?string $type;

    /**
     * The state of the fulfillment.
     * See [FulfillmentState](#type-fulfillmentstate) for possible values
     *
     * @var ?value-of<FulfillmentState> $state
     */
    #[JsonProperty('state')]
    private ?string $state;

    /**
     * Describes what order line items this fulfillment applies to.
     * It can be `ALL` or `ENTRY_LIST` with a supplied list of fulfillment entries.
     * See [FulfillmentFulfillmentLineItemApplication](#type-fulfillmentfulfillmentlineitemapplication) for possible values
     *
     * @var ?value-of<FulfillmentFulfillmentLineItemApplication> $lineItemApplication
     */
    #[JsonProperty('line_item_application')]
    private ?string $lineItemApplication;

    /**
     * A list of entries pertaining to the fulfillment of an order. Each entry must reference
     * a valid `uid` for an order line item in the `line_item_uid` field, as well as a `quantity` to
     * fulfill.
     *
     * Multiple entries can reference the same line item `uid`, as long as the total quantity among
     * all fulfillment entries referencing a single line item does not exceed the quantity of the
     * order's line item itself.
     *
     * An order cannot be marked as `COMPLETED` before all fulfillments are `COMPLETED`,
     * `CANCELED`, or `FAILED`. Fulfillments can be created and completed independently
     * before order completion.
     *
     * @var ?array<FulfillmentFulfillmentEntry> $entries
     */
    #[JsonProperty('entries'), ArrayType([FulfillmentFulfillmentEntry::class])]
    private ?array $entries;

    /**
     * Application-defined data attached to this fulfillment. Metadata fields are intended
     * to store descriptive references or associations with an entity in another system or store brief
     * information about the object. Square does not process this field; it only stores and returns it
     * in relevant API calls. Do not use metadata to store any sensitive information (such as personally
     * identifiable information or card details).
     *
     * Keys written by applications must be 60 characters or less and must be in the character set
     * `[a-zA-Z0-9_-]`. Entries can also include metadata generated by Square. These keys are prefixed
     * with a namespace, separated from the key with a ':' character.
     *
     * Values have a maximum length of 255 characters.
     *
     * An application can have up to 10 entries per metadata field.
     *
     * Entries written by applications are private and can only be read or modified by the same
     * application.
     *
     * For more information, see [Metadata](https://developer.squareup.com/docs/build-basics/metadata).
     *
     * @var ?array<string, ?string> $metadata
     */
    #[JsonProperty('metadata'), ArrayType(['string' => new Union('string', 'null')])]
    private ?array $metadata;

    /**
     * Contains details for a pickup fulfillment. These details are required when the fulfillment
     * type is `PICKUP`.
     *
     * @var ?FulfillmentPickupDetails $pickupDetails
     */
    #[JsonProperty('pickup_details')]
    private ?FulfillmentPickupDetails $pickupDetails;

    /**
     * Contains details for a shipment fulfillment. These details are required when the fulfillment type
     * is `SHIPMENT`.
     *
     * A shipment fulfillment's relationship to fulfillment `state`:
     * `PROPOSED`: A shipment is requested.
     * `RESERVED`: Fulfillment in progress. Shipment processing.
     * `PREPARED`: Shipment packaged. Shipping label created.
     * `COMPLETED`: Package has been shipped.
     * `CANCELED`: Shipment has been canceled.
     * `FAILED`: Shipment has failed.
     *
     * @var ?FulfillmentShipmentDetails $shipmentDetails
     */
    #[JsonProperty('shipment_details')]
    private ?FulfillmentShipmentDetails $shipmentDetails;

    /**
     * @var ?FulfillmentDeliveryDetails $deliveryDetails Describes delivery details of an order fulfillment.
     */
    #[JsonProperty('delivery_details')]
    private ?FulfillmentDeliveryDetails $deliveryDetails;

    /**
     * @param array{
     *   uid?: ?string,
     *   type?: ?value-of<FulfillmentType>,
     *   state?: ?value-of<FulfillmentState>,
     *   lineItemApplication?: ?value-of<FulfillmentFulfillmentLineItemApplication>,
     *   entries?: ?array<FulfillmentFulfillmentEntry>,
     *   metadata?: ?array<string, ?string>,
     *   pickupDetails?: ?FulfillmentPickupDetails,
     *   shipmentDetails?: ?FulfillmentShipmentDetails,
     *   deliveryDetails?: ?FulfillmentDeliveryDetails,
     * } $values
     */
    public function __construct(
        array $values = [],
    ) {
        $this->uid = $values['uid'] ?? null;
        $this->type = $values['type'] ?? null;
        $this->state = $values['state'] ?? null;
        $this->lineItemApplication = $values['lineItemApplication'] ?? null;
        $this->entries = $values['entries'] ?? null;
        $this->metadata = $values['metadata'] ?? null;
        $this->pickupDetails = $values['pickupDetails'] ?? null;
        $this->shipmentDetails = $values['shipmentDetails'] ?? null;
        $this->deliveryDetails = $values['deliveryDetails'] ?? null;
    }

    /**
     * @return ?string
     */
    public function getUid(): ?string
    {
        return $this->uid;
    }

    /**
     * @param ?string $value
     */
    public function setUid(?string $value = null): self
    {
        $this->uid = $value;
        return $this;
    }

    /**
     * @return ?value-of<FulfillmentType>
     */
    public function getType(): ?string
    {
        return $this->type;
    }

    /**
     * @param ?value-of<FulfillmentType> $value
     */
    public function setType(?string $value = null): self
    {
        $this->type = $value;
        return $this;
    }

    /**
     * @return ?value-of<FulfillmentState>
     */
    public function getState(): ?string
    {
        return $this->state;
    }

    /**
     * @param ?value-of<FulfillmentState> $value
     */
    public function setState(?string $value = null): self
    {
        $this->state = $value;
        return $this;
    }

    /**
     * @return ?value-of<FulfillmentFulfillmentLineItemApplication>
     */
    public function getLineItemApplication(): ?string
    {
        return $this->lineItemApplication;
    }

    /**
     * @param ?value-of<FulfillmentFulfillmentLineItemApplication> $value
     */
    public function setLineItemApplication(?string $value = null): self
    {
        $this->lineItemApplication = $value;
        return $this;
    }

    /**
     * @return ?array<FulfillmentFulfillmentEntry>
     */
    public function getEntries(): ?array
    {
        return $this->entries;
    }

    /**
     * @param ?array<FulfillmentFulfillmentEntry> $value
     */
    public function setEntries(?array $value = null): self
    {
        $this->entries = $value;
        return $this;
    }

    /**
     * @return ?array<string, ?string>
     */
    public function getMetadata(): ?array
    {
        return $this->metadata;
    }

    /**
     * @param ?array<string, ?string> $value
     */
    public function setMetadata(?array $value = null): self
    {
        $this->metadata = $value;
        return $this;
    }

    /**
     * @return ?FulfillmentPickupDetails
     */
    public function getPickupDetails(): ?FulfillmentPickupDetails
    {
        return $this->pickupDetails;
    }

    /**
     * @param ?FulfillmentPickupDetails $value
     */
    public function setPickupDetails(?FulfillmentPickupDetails $value = null): self
    {
        $this->pickupDetails = $value;
        return $this;
    }

    /**
     * @return ?FulfillmentShipmentDetails
     */
    public function getShipmentDetails(): ?FulfillmentShipmentDetails
    {
        return $this->shipmentDetails;
    }

    /**
     * @param ?FulfillmentShipmentDetails $value
     */
    public function setShipmentDetails(?FulfillmentShipmentDetails $value = null): self
    {
        $this->shipmentDetails = $value;
        return $this;
    }

    /**
     * @return ?FulfillmentDeliveryDetails
     */
    public function getDeliveryDetails(): ?FulfillmentDeliveryDetails
    {
        return $this->deliveryDetails;
    }

    /**
     * @param ?FulfillmentDeliveryDetails $value
     */
    public function setDeliveryDetails(?FulfillmentDeliveryDetails $value = null): self
    {
        $this->deliveryDetails = $value;
        return $this;
    }

    /**
     * @return string
     */
    public function __toString(): string
    {
        return $this->toJson();
    }
}
