<?php

namespace Square\Customers\Cards\Requests;

use Square\Core\Json\JsonSerializableType;
use Square\Core\Json\JsonProperty;
use Square\Types\Address;

class CreateCustomerCardRequest extends JsonSerializableType
{
    /**
     * @var string $customerId The Square ID of the customer profile the card is linked to.
     */
    private string $customerId;

    /**
     * A card nonce representing the credit card to link to the customer.
     *
     * Card nonces are generated by the Square payment form when customers enter
     * their card information. For more information, see
     * [Walkthrough: Integrate Square Payments in a Website](https://developer.squareup.com/docs/web-payments/take-card-payment).
     *
     * __NOTE:__ Card nonces generated by digital wallets (such as Apple Pay)
     * cannot be used to create a customer card.
     *
     * @var string $cardNonce
     */
    #[JsonProperty('card_nonce')]
    private string $cardNonce;

    /**
     * Address information for the card on file.
     *
     * __NOTE:__ If a billing address is provided in the request, the
     * `CreateCustomerCardRequest.billing_address.postal_code` must match
     * the postal code encoded in the card nonce.
     *
     * @var ?Address $billingAddress
     */
    #[JsonProperty('billing_address')]
    private ?Address $billingAddress;

    /**
     * @var ?string $cardholderName The full name printed on the credit card.
     */
    #[JsonProperty('cardholder_name')]
    private ?string $cardholderName;

    /**
     * An identifying token generated by [Payments.verifyBuyer()](https://developer.squareup.com/reference/sdks/web/payments/objects/Payments#Payments.verifyBuyer).
     * Verification tokens encapsulate customer device information and 3-D Secure
     * challenge results to indicate that Square has verified the buyer identity.
     *
     * @var ?string $verificationToken
     */
    #[JsonProperty('verification_token')]
    private ?string $verificationToken;

    /**
     * @param array{
     *   customerId: string,
     *   cardNonce: string,
     *   billingAddress?: ?Address,
     *   cardholderName?: ?string,
     *   verificationToken?: ?string,
     * } $values
     */
    public function __construct(
        array $values,
    ) {
        $this->customerId = $values['customerId'];
        $this->cardNonce = $values['cardNonce'];
        $this->billingAddress = $values['billingAddress'] ?? null;
        $this->cardholderName = $values['cardholderName'] ?? null;
        $this->verificationToken = $values['verificationToken'] ?? null;
    }

    /**
     * @return string
     */
    public function getCustomerId(): string
    {
        return $this->customerId;
    }

    /**
     * @param string $value
     */
    public function setCustomerId(string $value): self
    {
        $this->customerId = $value;
        return $this;
    }

    /**
     * @return string
     */
    public function getCardNonce(): string
    {
        return $this->cardNonce;
    }

    /**
     * @param string $value
     */
    public function setCardNonce(string $value): self
    {
        $this->cardNonce = $value;
        return $this;
    }

    /**
     * @return ?Address
     */
    public function getBillingAddress(): ?Address
    {
        return $this->billingAddress;
    }

    /**
     * @param ?Address $value
     */
    public function setBillingAddress(?Address $value = null): self
    {
        $this->billingAddress = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getCardholderName(): ?string
    {
        return $this->cardholderName;
    }

    /**
     * @param ?string $value
     */
    public function setCardholderName(?string $value = null): self
    {
        $this->cardholderName = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getVerificationToken(): ?string
    {
        return $this->verificationToken;
    }

    /**
     * @param ?string $value
     */
    public function setVerificationToken(?string $value = null): self
    {
        $this->verificationToken = $value;
        return $this;
    }
}
