<?php
/**
 * General Rate Limiting module.
 *
 * @package   Charitable/Functions/Admin
 * @author    David Bisset
 * @copyright Copyright (c) 2023, WP Charitable LLC
 * @license   http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     1.8.9
 * @version   1.8.9
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Charitable_General_Rate_Limiting' ) ) :

	/**
	 * General Rate Limiting module.
	 *
	 * @since 1.8.9
	 */
	class Charitable_General_Rate_Limiting {

		/**
		 * Is the file writable.
		 *
		 * @var bool
		 */
		public $is_writable = true;

		/**
		 * Set up the charitable Rate Limiting Class
		 *
		 * @since  1.8.9
		 */
		public function __construct() {
			if ( $this->is_active() ) {
				$this->setup();
			}
		}

		/**
		 * Return whether the module is active.
		 *
		 * @since  1.8.9
		 *
		 * @return boolean
		 */
		public function is_active() {
			return 'disabled' !== charitable_get_option( 'general_rate_limiting_status', 'enabled' );
		}

		/**
		 * Set up module hooks.
		 *
		 * @since  1.8.9
		 *
		 * @return void
		 */
		public function setup() {
			// Schedule cleanup if not already scheduled.
			add_action( 'init', array( $this, 'schedule_cleanup' ) );

			// Hook into the scheduled cleanup.
			add_action( 'charitable_cleanup_general_rate_limiting_log', array( $this, 'cleanup_log' ) );

			// Validate donation form submissions.
			add_filter( 'charitable_validate_donation_form_submission_security_check', array( $this, 'validate_rate_limit' ), 5, 2 );

			// Check before processing other forms.
			add_action( 'charitable_retrieve_password', array( $this, 'check_rate_limit_before_form_processing' ), 1 );
			add_action( 'charitable_reset_password', array( $this, 'check_rate_limit_before_form_processing' ), 1 );
			add_action( 'charitable_update_profile', array( $this, 'check_rate_limit_before_form_processing' ), 1 );
			add_action( 'charitable_save_registration', array( $this, 'check_rate_limit_before_form_processing' ), 1 );

			// Clear log file when setting is saved.
			add_filter( 'charitable_save_settings', array( $this, 'clear_log_file' ), 10, 2 );

			// Validate blocking period setting.
			add_filter( 'charitable_save_settings', array( $this, 'validate_blocking_period' ), 5, 2 );

			// Validate max requests and time window settings.
			add_filter( 'charitable_save_settings', array( $this, 'validate_rate_limiting_settings' ), 5, 2 );
		}

		/**
		 * Get the module settings.
		 *
		 * @since  1.8.9
		 *
		 * @return array
		 */
		public function get_settings() {
			return array(
				'rl_general_header'               => array(
					'title'    => __( 'General Rate Limiting', 'charitable' ),
					'type'     => 'heading',
					'class'    => 'section-heading',
					'priority' => 230,
				),
				'rl_general_stats'                => array(
					'title'    => __( 'Rate Limiting Statistics', 'charitable' ),
					'type'     => 'text',
					'callback' => array( $this, 'render_stats_summary' ),
					'priority' => 232,
				),
				'general_rate_limiting_status'    => array(
					'title'    => __( 'Status', 'charitable' ),
					'type'     => 'select',
					'options'  => array(
						'disabled' => __( 'Disabled', 'charitable' ),
						'enabled'  => __( 'Enabled', 'charitable' ),
					),
					'priority' => 234,
					'default'  => 'enabled',
					'help'     => __( 'Protects against form spam and abuse by tracking all form submissions (donations, registrations, password resets, etc.) by IP address.', 'charitable' ),
				),
				'rl_general_max_requests'         => array(
					'title'    => __( 'Max Requests', 'charitable' ),
					'type'     => 'number',
					'help'     => __( 'The maximum number of form submissions allowed within the time window before blocking.', 'charitable' ),
					'priority' => 238,
					'default'  => 5,
					'attrs'    => array(
						'data-trigger-key'   => '#charitable_settings_general_rate_limiting_status',
						'data-trigger-value' => '!disabled',
						'min'                => '1',
					),
				),
				'rl_general_time_window'          => array(
					'title'    => __( 'Time Window (hours)', 'charitable' ),
					'type'     => 'number',
					'help'     => __( 'The time window in hours for counting requests. Default: 1 hour.', 'charitable' ),
					'priority' => 242,
					'default'  => 1,
					'attrs'    => array(
						'data-trigger-key'   => '#charitable_settings_general_rate_limiting_status',
						'data-trigger-value' => '!disabled',
						'min'                => '0.5',
						'step'               => '0.5',
					),
				),
				'rl_general_blocking_period'       => array(
					'title'    => __( 'Blocking Period (hours)', 'charitable' ),
					'type'     => 'number',
					'help'     => __( 'The length of time (in hours) an IP address is blocked after exceeding the rate limit. Default: 2.5 hours.', 'charitable' ),
					'priority' => 246,
					'default'  => 2.5,
					'attrs'    => array(
						'data-trigger-key'   => '#charitable_settings_general_rate_limiting_status',
						'data-trigger-value' => '!disabled',
						'step'               => '0.5',
						'min'                => '0.5',
					),
				),
				'rl_general_clear_log_file'       => array(
					'label_for' => __( 'Clear Log File', 'charitable' ),
					'type'      => 'checkbox',
					'help'      => $this->get_clear_log_help_text(),
					'priority'  => 250,
					'attrs'     => array(
						'data-trigger-key'   => '#charitable_settings_general_rate_limiting_status',
						'data-trigger-value' => '!disabled',
					),
				),
			);
		}

		/**
		 * Schedules a cleanup of the rate limit log entries.
		 *
		 * Runs every hour, and clears any rate limiting logs that are past expiration.
		 *
		 * @since  1.8.9
		 *
		 * @return void
		 */
		public function schedule_cleanup() {
			if ( ! wp_next_scheduled( 'charitable_cleanup_general_rate_limiting_log' ) ) {
				wp_schedule_event(
					time(),
					'hourly',
					'charitable_cleanup_general_rate_limiting_log'
				);
			}
		}

		/**
		 * Removes expired entries from the rate limit log (triggered by cron).
		 *
		 * @since  1.8.9
		 *
		 * @return void
		 */
		public function cleanup_log() {
			$current_logs = $this->get_decoded_file();

			if ( empty( $current_logs ) ) {
				if ( charitable_is_debug() ) {
					error_log( '[Charitable Rate Limiting] Cleanup: No log entries to clean.' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				}
				return;
			}

			$cleaned_count = 0;
			$before_count = count( $current_logs );

			foreach ( $current_logs as $blocking_id => $entry ) {
				$expiration = ! empty( $entry['timeout'] ) ? $entry['timeout'] : 0;

				if ( $expiration < time() ) {
					if ( charitable_is_debug() ) {
						error_log( '[Charitable Rate Limiting] Cleanup: Removing expired entry for IP ' . $blocking_id . ' (expired: ' . gmdate( 'Y-m-d H:i:s', $expiration ) . ')' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
					}
					unset( $current_logs[ $blocking_id ] );
					$cleaned_count++;
				}
			}

			if ( $cleaned_count > 0 ) {
				$this->write_to_log( $current_logs );
				if ( charitable_is_debug() ) {
					error_log( '[Charitable Rate Limiting] Cleanup: Removed ' . $cleaned_count . ' expired entries. Remaining: ' . count( $current_logs ) . ' (was: ' . $before_count . ')' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				}
			} elseif ( charitable_is_debug() ) {
				error_log( '[Charitable Rate Limiting] Cleanup: No expired entries found. Total entries: ' . $before_count ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}
		}

		/**
		 * Checks if the current IP address has hit the rate limit.
		 *
		 * @since  1.8.9
		 *
		 * @return bool
		 */
		public function has_hit_rate_limit() {
			if ( ! $this->is_active() ) {
				if ( charitable_is_debug() ) {
					error_log( '[Charitable Rate Limiting] Rate limiting is disabled.' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				}
				return false;
			}

			// Check IP whitelist/blacklist if spam blocker is active.
			$ip_manager = $this->get_ip_manager();
			if ( $ip_manager ) {
				$current_ip = $this->get_rate_limit_id();

				// Whitelisted IPs bypass rate limiting.
				if ( $ip_manager->is_whitelisted( $current_ip ) ) {
					if ( charitable_is_debug() ) {
						error_log( '[Charitable Rate Limiting] IP ' . $current_ip . ' is whitelisted, bypassing rate limit.' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
					}
					return false;
				}

				// Blacklisted IPs are always blocked.
				if ( $ip_manager->is_blacklisted( $current_ip ) ) {
					if ( charitable_is_debug() ) {
						error_log( '[Charitable Rate Limiting] IP ' . $current_ip . ' is blacklisted, blocking request.' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
					}
					return true;
				}
			}

			$blocking_id = $this->get_rate_limit_id();
			$entry       = $this->get_rate_limiting_entry( $blocking_id );
			$expiration  = ! empty( $entry['timeout'] ) ? $entry['timeout'] : 0;
			$count       = ! empty( $entry['count'] ) ? $entry['count'] : 0;

			// Previous request limit expiration has passed. Start fresh.
			if ( $expiration < time() ) {
				if ( charitable_is_debug() ) {
					error_log( '[Charitable Rate Limiting] Entry expired for IP ' . $blocking_id . ' (count was: ' . $count . '). Starting fresh.' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				}
				$this->remove_log_entry( $blocking_id );
				return false;
			}

			$max_requests = intval( charitable_get_option( 'rl_general_max_requests', 5 ) );

			/**
			 * Filters the maximum number of requests allowed within the time window.
			 *
			 * @since  1.8.9
			 *
			 * @param int $max_requests The maximum number of requests. Default 5.
			 */
			$max_requests = apply_filters( 'charitable_general_rate_limiting_max_requests', $max_requests );

			$has_hit_limit = $count >= $max_requests;

			if ( charitable_is_debug() ) {
				error_log( '[Charitable Rate Limiting] IP: ' . $blocking_id . ' | Count: ' . $count . '/' . $max_requests . ' | Expires: ' . gmdate( 'Y-m-d H:i:s', $expiration ) . ' | Hit Limit: ' . ( $has_hit_limit ? 'YES' : 'NO' ) ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}

			return $has_hit_limit;
		}

		/**
		 * Validate rate limit on donation form submission.
		 *
		 * @since  1.8.9
		 *
		 * @param  boolean                  $ret  The result to be returned. True or False.
		 * @param  Charitable_Donation_Form $form The donation form object.
		 * @return boolean
		 */
		public function validate_rate_limit( $ret, Charitable_Donation_Form $form ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
			if ( ! $ret ) {
				return $ret;
			}

			// Check IP blacklist first (before rate limiting check) if spam blocker is active.
			$ip_manager = $this->get_ip_manager();
			if ( $ip_manager ) {
				$current_ip = $this->get_rate_limit_id();

				// Blacklisted IPs are always blocked.
				if ( $ip_manager->is_blacklisted( $current_ip ) ) {
					charitable_get_notices()->add_error( __( 'Your IP address has been blocked. Please contact support if you believe this is an error.', 'charitable' ) );
					return false;
				}

				// Whitelisted IPs bypass rate limiting.
				if ( $ip_manager->is_whitelisted( $current_ip ) ) {
					return $ret;
				}
			}

			if ( $this->has_hit_rate_limit() ) {
				$blocking_period = floatval( charitable_get_option( 'rl_general_blocking_period', 2.5 ) );
				if ( charitable_is_debug() ) {
					error_log( '[Charitable Rate Limiting] Donation form submission BLOCKED due to rate limit exceeded.' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				}

				$error_message = sprintf(
					/* translators: %s: blocking period in hours */
					__( 'You have exceeded the rate limit. Please try again in %s hours.', 'charitable' ),
					number_format( $blocking_period, 1 )
				);

				// Add link to security settings if in test mode and user is logged in.
				if ( $this->is_test_mode() && is_user_logged_in() ) {
					$security_url = admin_url( 'admin.php?page=charitable-settings&tab=security' );
					$error_message .= ' <a href="' . esc_url( $security_url ) . '">' . __( 'If you are testing temporarily disable rate limiting.', 'charitable' ) . '</a>';
				}

				charitable_get_notices()->add_error( $error_message );
				return false;
			}

			// Increment the count for this IP.
			$new_count = $this->increment_rate_limit_count();
			if ( charitable_is_debug() ) {
				error_log( '[Charitable Rate Limiting] Donation form submission allowed. Rate limit count incremented to: ' . $new_count ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}

			return $ret;
		}

		/**
		 * Get IP manager instance (if spam blocker is active).
		 *
		 * @since  1.8.9
		 *
		 * @return Charitable_Spam_Blocker_IP_Management|null
		 */
		protected function get_ip_manager() {
			if ( class_exists( 'Charitable_Spam_Blocker_IP_Management' ) ) {
				static $ip_manager = null;
				if ( is_null( $ip_manager ) ) {
					$ip_manager = new Charitable_Spam_Blocker_IP_Management();
				}
				return $ip_manager;
			}
			return null;
		}

		/**
		 * Check rate limit before processing a form submission.
		 *
		 * If the rate limit check fails, form processing is blocked.
		 *
		 * @since  1.8.9
		 *
		 * @return void
		 */
		public function check_rate_limit_before_form_processing() {
			// Check IP blacklist first if spam blocker is active.
			$ip_manager = $this->get_ip_manager();
			if ( $ip_manager ) {
				$current_ip = $this->get_rate_limit_id();

				// Blacklisted IPs are always blocked.
				if ( $ip_manager->is_blacklisted( $current_ip ) ) {
					$form_key = $this->get_current_form_from_hook();

					switch ( $form_key ) {
						case 'registration_form':
							remove_action( 'charitable_save_registration', array( 'Charitable_Registration_Form', 'save_registration' ) );
							break;

						case 'profile_form':
							remove_action( 'charitable_update_profile', array( 'Charitable_Profile_Form', 'update_profile' ) );
							break;

						case 'password_retrieval_form':
							remove_action( 'charitable_retrieve_password', array( 'Charitable_Forgot_Password_Form', 'retrieve_password' ) );
							break;

						case 'password_reset_form':
							remove_action( 'charitable_reset_password', array( 'Charitable_Reset_Password_Form', 'reset_password' ) );
							break;

						case 'campaign_form':
							remove_action( 'charitable_save_campaign', array( 'Charitable_Ambassadors_Campaign_Form', 'save_campaign' ) );
							break;
					}

					charitable_get_notices()->add_error( __( 'Your IP address has been blocked. Please contact support if you believe this is an error.', 'charitable' ) );
					return;
				}

				// Whitelisted IPs bypass rate limiting.
				if ( $ip_manager->is_whitelisted( $current_ip ) ) {
					return;
				}
			}

					if ( $this->has_hit_rate_limit() ) {
						$form_key = $this->get_current_form_from_hook();
						if ( charitable_is_debug() ) {
							error_log( '[Charitable Rate Limiting] Form submission BLOCKED: ' . $form_key . ' (rate limit exceeded)' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
						}

						switch ( $form_key ) {
							case 'registration_form':
								remove_action( 'charitable_save_registration', array( 'Charitable_Registration_Form', 'save_registration' ) );
								break;

							case 'profile_form':
								remove_action( 'charitable_update_profile', array( 'Charitable_Profile_Form', 'update_profile' ) );
								break;

							case 'password_retrieval_form':
								remove_action( 'charitable_retrieve_password', array( 'Charitable_Forgot_Password_Form', 'retrieve_password' ) );
								break;

							case 'password_reset_form':
								remove_action( 'charitable_reset_password', array( 'Charitable_Reset_Password_Form', 'reset_password' ) );
								break;

							case 'campaign_form':
								remove_action( 'charitable_save_campaign', array( 'Charitable_Ambassadors_Campaign_Form', 'save_campaign' ) );
								break;
						}

						$blocking_period = floatval( charitable_get_option( 'rl_general_blocking_period', 2.5 ) );

						$error_message = sprintf(
							/* translators: %s: blocking period in hours */
							__( 'You have exceeded the rate limit. Please try again in %s hours.', 'charitable' ),
							number_format( $blocking_period, 1 )
						);

						// Add link to security settings if in test mode and user is logged in.
						if ( $this->is_test_mode() && is_user_logged_in() ) {
							$security_url = admin_url( 'admin.php?page=charitable-settings&tab=security' );
							$error_message .= ' <a href="' . esc_url( $security_url ) . '">' . __( 'If you are testing temporarily disable rate limiting.', 'charitable' ) . '</a>';
						}

						charitable_get_notices()->add_error( $error_message );
					} else {
						// Increment the count for this IP.
						$new_count = $this->increment_rate_limit_count();
						$form_key = $this->get_current_form_from_hook();
						if ( charitable_is_debug() ) {
							error_log( '[Charitable Rate Limiting] Form submission allowed: ' . $form_key . ' | Rate limit count: ' . $new_count ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
						}
					}
		}

		/**
		 * Return the form key based on the hook.
		 *
		 * @since  1.8.9
		 *
		 * @return string
		 */
		public function get_current_form_from_hook() {
			switch ( current_filter() ) {
				case 'charitable_save_registration':
					$form_key = 'registration_form';
					break;

				case 'charitable_update_profile':
					$form_key = 'profile_form';
					break;

				case 'charitable_retrieve_password':
					$form_key = 'password_retrieval_form';
					break;

				case 'charitable_reset_password':
					$form_key = 'password_reset_form';
					break;

				case 'charitable_save_campaign':
					$form_key = 'campaign_form';
					break;

				default:
					$form_key = null;
			}
			return $form_key;
		}

		/**
		 * Remove an entry from the rate limiting log.
		 *
		 * @since  1.8.9
		 *
		 * @param string $blocking_id The blocking ID for the rate limiting.
		 */
		public function remove_log_entry( $blocking_id = '' ) {
			$current_logs = $this->get_decoded_file();
			unset( $current_logs[ $blocking_id ] );

			$this->write_to_log( $current_logs );
		}

		/**
		 * Get a specific entry from the rate limiting log.
		 *
		 * @since  1.8.9
		 *
		 * @param string $blocking_id The blocking ID to get the entry for.
		 *
		 * @return array
		 */
		public function get_rate_limiting_entry( $blocking_id = '' ) {
			$current_logs = $this->get_decoded_file();
			$entry        = array();

			if ( array_key_exists( $blocking_id, $current_logs ) ) {
				$entry = $current_logs[ $blocking_id ];
			}

			return $entry;
		}

		/**
		 * Retrieves the number of times an IP address has made requests.
		 *
		 * @since  1.8.9
		 *
		 * @return int
		 */
		public function get_rate_count() {
			$blocking_id = $this->get_rate_limit_id();
			$count       = 0;

			$current_blocks = $this->get_decoded_file();
			if ( array_key_exists( $blocking_id, $current_blocks ) ) {
				$count = $current_blocks[ $blocking_id ]['count'];
			}

			return $count;
		}

		/**
		 * Increments the rate limit counter.
		 *
		 * @since  1.8.9
		 *
		 * @return int
		 */
		public function increment_rate_limit_count() {
			$current_count = $this->get_rate_count();
			$blocking_id   = $this->get_rate_limit_id();

			if ( empty( $current_count ) ) {
				$current_count = 1;
			} else {
				++$current_count;
			}

			$this->update_rate_limiting_count( $blocking_id, $current_count );

			if ( charitable_is_debug() ) {
				error_log( '[Charitable Rate Limiting] Incremented count for IP ' . $blocking_id . ' to ' . $current_count ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}

			return absint( $current_count );
		}

		/**
		 * Update an entry in the rate limiting array.
		 *
		 * @since  1.8.9
		 *
		 * @param string $blocking_id   The blocking ID.
		 * @param int    $current_count The count to update to.
		 */
		protected function update_rate_limiting_count( $blocking_id = '', $current_count = 0 ) {
			$time_window_hours = floatval( charitable_get_option( 'rl_general_time_window', 1 ) );
			$blocking_period_hours = floatval( charitable_get_option( 'rl_general_blocking_period', 2.5 ) );

			// Calculate expiration: time window for counting, then blocking period after limit is hit.
			$expiration_in_seconds = ( $time_window_hours * HOUR_IN_SECONDS ) + ( $blocking_period_hours * HOUR_IN_SECONDS );

			/**
			 * Filters the length of time before rate limits are reset.
			 *
			 * @since  1.8.9
			 *
			 * @param int $expiration_in_seconds The length in seconds before rate limit counts are reset.
			 */
			$expiration_in_seconds = apply_filters( 'charitable_general_rate_limiting_timeout', $expiration_in_seconds );

			$current_log = $this->get_decoded_file();

			// New entry, add expiration time.
			$is_new_entry = ! isset( $current_log[ $blocking_id ] );
			if ( $is_new_entry ) {
				$current_log[ $blocking_id ]['timeout'] = time() + $expiration_in_seconds;
				if ( charitable_is_debug() ) {
					error_log( '[Charitable Rate Limiting] Created new entry for IP ' . $blocking_id . ' | Timeout: ' . gmdate( 'Y-m-d H:i:s', $current_log[ $blocking_id ]['timeout'] ) ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				}
			}

			// Always increment count.
			$current_log[ $blocking_id ]['count'] = $current_count;

			// If we've hit the limit, extend the timeout to include blocking period.
			$max_requests = intval( charitable_get_option( 'rl_general_max_requests', 5 ) );
			if ( $current_count >= $max_requests ) {
				$blocking_period_seconds = $blocking_period_hours * HOUR_IN_SECONDS;
				$current_log[ $blocking_id ]['timeout'] = time() + $blocking_period_seconds;
				if ( charitable_is_debug() ) {
					error_log( '[Charitable Rate Limiting] Rate limit EXCEEDED for IP ' . $blocking_id . ' | Blocked until: ' . gmdate( 'Y-m-d H:i:s', $current_log[ $blocking_id ]['timeout'] ) ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				}
			}

			$this->write_to_log( $current_log );
		}

		/**
		 * Generates the rate limiting tracking ID.
		 *
		 * ID is the IP address of the visitor.
		 *
		 * @since  1.8.9
		 *
		 * @return string
		 */
		public function get_rate_limit_id() {
			// Use charitable_get_ip if available (from core).
			if ( function_exists( 'charitable_get_ip' ) ) {
				return charitable_get_ip();
			}

			// Use spam blocker function if available.
			if ( function_exists( 'charitable_spam_buster_get_ip' ) ) {
				return charitable_spam_buster_get_ip();
			}

			// Final fallback.
			return isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : ''; // phpcs:ignore
		}

		/**
		 * Validate blocking period setting.
		 *
		 * Ensures blocking period is in 0.5 hour increments and minimum 0.5 hours.
		 *
		 * @since  1.8.9
		 *
		 * @param  array $values     Current settings values.
		 * @param  array $new_values New settings values being saved.
		 * @return array
		 */
		public function validate_blocking_period( $values, $new_values ) {
			if ( ! isset( $new_values['rl_general_blocking_period'] ) ) {
				return $values;
			}

			$blocking_period = floatval( $new_values['rl_general_blocking_period'] );

			// Ensure minimum of 0.5 hours.
			if ( $blocking_period < 0.5 ) {
				$blocking_period = 0.5;
				charitable_get_admin_notices()->add_notice(
					__( 'Blocking Period must be at least 0.5 hours. Value has been adjusted.', 'charitable' ),
					'warning',
					false,
					true
				);
			}

			// Round to nearest 0.5 hour increment.
			$blocking_period = round( $blocking_period * 2 ) / 2;

			$values['rl_general_blocking_period'] = $blocking_period;

			return $values;
		}

		/**
		 * Validate max requests and time window settings.
		 *
		 * Ensures max requests and time window are at least 1.
		 *
		 * @since  1.8.9
		 *
		 * @param  array $values     Current settings values.
		 * @param  array $new_values New settings values being saved.
		 * @return array
		 */
		public function validate_rate_limiting_settings( $values, $new_values ) {
			// Validate max requests.
			if ( isset( $new_values['rl_general_max_requests'] ) ) {
				$max_requests = intval( $new_values['rl_general_max_requests'] );
				if ( $max_requests < 1 ) {
					$max_requests = 1;
					charitable_get_admin_notices()->add_notice(
						__( 'Max Requests must be at least 1. Value has been adjusted.', 'charitable' ),
						'warning',
						false,
						true
					);
				}
				$values['rl_general_max_requests'] = $max_requests;
			}

			// Validate time window.
			if ( isset( $new_values['rl_general_time_window'] ) ) {
				$time_window = floatval( $new_values['rl_general_time_window'] );
				if ( $time_window < 0.5 ) {
					$time_window = 0.5;
					charitable_get_admin_notices()->add_notice(
						__( 'Time Window must be at least 0.5 hours. Value has been adjusted.', 'charitable' ),
						'warning',
						false,
						true
					);
				}
				// Round to nearest 0.5 hour increment.
				$time_window = round( $time_window * 2 ) / 2;
				$values['rl_general_time_window'] = $time_window;
			}

			return $values;
		}

		/**
		 * Clear log file when setting is saved.
		 *
		 * @since  1.8.9
		 *
		 * @param  array $values     Current settings values.
		 * @param  array $new_values New settings values being saved.
		 * @return array
		 */
		public function clear_log_file( $values, $new_values ) {

			/* If this option isn't in the return values or isn't checked off, leave. */
			if ( ! isset( $new_values['rl_general_clear_log_file'] ) || 0 === intval( $new_values['rl_general_clear_log_file'] ) ) {
				return $values;
			}

			$log_path = $this->get_log_file_path();

			if ( @file_exists( $log_path ) ) { // phpcs:ignore
				@unlink( $log_path ); // phpcs:ignore
			}

			// Allow an addon to hook into this.
			do_action( 'charitable_clear_general_rate_limiting_log_file' );

			charitable_get_admin_notices()->add_notice( __( 'Charitable rate limiting log file has been cleared.', 'charitable' ), 'success', false, true );

			$values['rl_general_clear_log_file'] = false;

			return $values;
		}

		/**
		 * Get the decoded array of rate limiting from the log file.
		 *
		 * @since  1.8.9
		 *
		 * @return array
		 */
		protected function get_decoded_file() {
			$decoded_contents = json_decode( $this->get_file_contents(), true );
			if ( is_null( $decoded_contents ) ) {
				$decoded_contents = array();
			}

			return (array) $decoded_contents;
		}

		/**
		 * Retrieve the log data
		 *
		 * @since  1.8.9
		 * @return string
		 */
		protected function get_file_contents() {
			return $this->get_file();
		}

		/**
		 * Retrieve the file data is written to
		 *
		 * @since  1.8.9
		 * @return string
		 */
		protected function get_file() {

			$file = wp_json_encode( array() );
			$file_with_path = $this->get_log_file_path();

			if ( empty( $file_with_path ) ) {
				return $file;
			}

			if ( @file_exists( $file_with_path ) ) { // phpcs:ignore

				$file = @file_get_contents( $file_with_path ); // phpcs:ignore
			} else {

				// Ensure directory exists.
				$upload_dir = wp_upload_dir();
				$rate_limiting_dir = trailingslashit( $upload_dir['basedir'] ) . 'charitable/rate-limiting/';

				if ( ! file_exists( $rate_limiting_dir ) ) {
					wp_mkdir_p( $rate_limiting_dir );
				}

				@file_put_contents( $file_with_path, $file ); // phpcs:ignore
				@chmod( $file_with_path, 0664 ); // phpcs:ignore
			}

			return $file;
		}

		/**
		 * Get the log file path.
		 *
		 * @since  1.8.9
		 *
		 * @return string
		 */
		protected function get_log_file_path() {
			$upload_dir = wp_upload_dir();
			$rate_limiting_dir = trailingslashit( $upload_dir['basedir'] ) . 'charitable/rate-limiting/';

			// Ensure directory exists.
			if ( ! file_exists( $rate_limiting_dir ) ) {
				wp_mkdir_p( $rate_limiting_dir );
			}

			$filename = wp_hash( home_url( '/' ) ) . '-charitable-general-rate-limiting.log';

			return trailingslashit( $rate_limiting_dir ) . $filename;
		}

		/**
		 * Get the log file URL.
		 *
		 * @since  1.8.9
		 *
		 * @return string
		 */
		protected function get_log_file_url() {
			$upload_dir = wp_upload_dir();
			$rate_limiting_dir = trailingslashit( $upload_dir['baseurl'] ) . 'charitable/rate-limiting/';

			$filename = wp_hash( home_url( '/' ) ) . '-charitable-general-rate-limiting.log';

			return trailingslashit( $rate_limiting_dir ) . $filename;
		}

		/**
		 * Write the log message
		 *
		 * @since  1.8.9
		 *
		 * @param array $content The content of the rate limiting.
		 *
		 * @return void
		 */
		public function write_to_log( $content = array() ) {
			$content = wp_json_encode( $content );

			if ( $this->is_log_writable() ) {
				$file_path = $this->get_log_file_path();
				@file_put_contents( $file_path, $content ); // phpcs:ignore
			}
		}

		/**
		 * Check if log file is writable.
		 *
		 * @since  1.8.9
		 *
		 * @return bool
		 */
		protected function is_log_writable() {
			$upload_dir = wp_upload_dir();
			return is_writeable( $upload_dir['basedir'] ); // phpcs:ignore
		}

		/**
		 * Render rate limiting statistics summary.
		 *
		 * @since  1.8.9
		 *
		 * @return void
		 */
		public function render_stats_summary() {
			$stats = $this->get_rate_limiting_stats();

			ob_start();
			?>
			<div class="charitable-rate-limiting-stats" style="background: #f9f9f9; border: 1px solid #ddd; border-radius: 4px; padding: 15px; margin: 10px 0;">
				<h4 style="margin-top: 0;"><?php esc_html_e( 'Current Status', 'charitable' ); ?></h4>
				<table class="widefat" style="margin: 0;">
					<tbody>
						<tr>
							<td style="width: 50%; padding: 8px;"><strong><?php esc_html_e( 'IPs Currently Tracked:', 'charitable' ); ?></strong></td>
							<td style="padding: 8px;"><?php echo esc_html( $stats['total_ips'] ); ?></td>
						</tr>
						<tr>
							<td style="padding: 8px;"><strong><?php esc_html_e( 'IPs Currently Blocked:', 'charitable' ); ?></strong></td>
							<td style="padding: 8px;">
								<span style="color: <?php echo esc_attr( $stats['blocked_ips'] > 0 ? '#d63638' : '#00a32a' ); ?>;">
									<?php echo esc_html( $stats['blocked_ips'] ); ?>
								</span>
							</td>
						</tr>
						<tr>
							<td style="padding: 8px;"><strong><?php esc_html_e( 'Total Requests Tracked:', 'charitable' ); ?></strong></td>
							<td style="padding: 8px;"><?php echo esc_html( $stats['total_requests'] ); ?></td>
						</tr>
						<tr>
							<td style="padding: 8px;"><strong><?php esc_html_e( 'Average Requests per IP:', 'charitable' ); ?></strong></td>
							<td style="padding: 8px;"><?php echo esc_html( $stats['avg_requests'] ); ?></td>
						</tr>
					</tbody>
				</table>
				<?php if ( ! empty( $stats['recent_ips'] ) ) : ?>
					<div style="margin-top: 15px;">
						<h4 style="margin-bottom: 8px;"><?php esc_html_e( 'Recently Active IPs', 'charitable' ); ?></h4>
						<table class="widefat" style="margin: 0;">
							<thead>
								<tr>
									<th style="padding: 8px;"><?php esc_html_e( 'IP Address', 'charitable' ); ?></th>
									<th style="padding: 8px;"><?php esc_html_e( 'Requests', 'charitable' ); ?></th>
									<th style="padding: 8px;"><?php esc_html_e( 'Status', 'charitable' ); ?></th>
									<th style="padding: 8px;"><?php esc_html_e( 'Expires', 'charitable' ); ?></th>
								</tr>
							</thead>
							<tbody>
								<?php foreach ( $stats['recent_ips'] as $ip_data ) : ?>
									<tr>
										<td style="padding: 8px; font-family: monospace;"><?php echo esc_html( $ip_data['ip'] ); ?></td>
										<td style="padding: 8px;"><?php echo esc_html( $ip_data['count'] ); ?></td>
										<td style="padding: 8px;">
											<?php if ( $ip_data['blocked'] ) : ?>
												<span style="color: #d63638; font-weight: bold;"><?php esc_html_e( 'Blocked', 'charitable' ); ?></span>
											<?php else : ?>
												<span style="color: #00a32a;"><?php esc_html_e( 'Active', 'charitable' ); ?></span>
											<?php endif; ?>
										</td>
										<td style="padding: 8px;">
											<?php
											if ( $ip_data['expires'] > 0 ) {
												$time_remaining = $ip_data['expires'] - time();
												if ( $time_remaining > 0 ) {
													$hours   = floor( $time_remaining / HOUR_IN_SECONDS );
													$minutes = floor( ( $time_remaining % HOUR_IN_SECONDS ) / 60 );
													if ( $hours > 0 ) {
														/* translators: %d: number of hours */
														echo esc_html( sprintf( _n( '%d hour', '%d hours', $hours, 'charitable' ), $hours ) );
														if ( $minutes > 0 ) {
															/* translators: %d: number of minutes */
															echo ' ' . esc_html( sprintf( _n( '%d minute', '%d minutes', $minutes, 'charitable' ), $minutes ) );
														}
													} else {
														/* translators: %d: number of minutes */
														echo esc_html( sprintf( _n( '%d minute', '%d minutes', $minutes, 'charitable' ), $minutes ) );
													}
												} else {
													esc_html_e( 'Expired', 'charitable' );
												}
											} else {
												esc_html_e( 'N/A', 'charitable' );
											}
											?>
										</td>
									</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
					</div>
				<?php endif; ?>
			</div>
			<?php
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- HTML is safe, generated internally.
			echo ob_get_clean();
		}

		/**
		 * Get rate limiting statistics.
		 *
		 * @since  1.8.9
		 *
		 * @return array
		 */
		protected function get_rate_limiting_stats() {
			$current_logs = $this->get_decoded_file();
			$max_requests = intval( charitable_get_option( 'rl_general_max_requests', 5 ) );

			$stats = array(
				'total_ips'      => 0,
				'blocked_ips'    => 0,
				'total_requests' => 0,
				'avg_requests'   => 0,
				'recent_ips'     => array(),
			);

			if ( empty( $current_logs ) ) {
				return $stats;
			}

			$now = time();
			$active_ips = array();

			foreach ( $current_logs as $ip => $entry ) {
				$expiration = ! empty( $entry['timeout'] ) ? $entry['timeout'] : 0;
				$count = ! empty( $entry['count'] ) ? $entry['count'] : 0;

				// Skip expired entries.
				if ( $expiration < $now ) {
					continue;
				}

				$stats['total_ips']++;
				$stats['total_requests'] += $count;

				$is_blocked = $count >= $max_requests;
				if ( $is_blocked ) {
					$stats['blocked_ips']++;
				}

				$active_ips[] = array(
					'ip'       => $ip,
					'count'    => $count,
					'blocked'  => $is_blocked,
					'expires'  => $expiration,
				);
			}

			// Calculate average.
			if ( $stats['total_ips'] > 0 ) {
				$stats['avg_requests'] = round( $stats['total_requests'] / $stats['total_ips'], 2 );
			}

			// Sort by count (descending) and get top 10.
			usort(
				$active_ips,
				function( $a, $b ) {
					return $b['count'] - $a['count'];
				}
			);

			$stats['recent_ips'] = array_slice( $active_ips, 0, 10 );

			return $stats;
		}

		/**
		 * Get help text for clear log file setting with link to log file.
		 *
		 * @since  1.8.9
		 *
		 * @return string
		 */
		protected function get_clear_log_help_text() {
			$help_text = __( 'Clear all tracking of currently logged rate limit attempts.', 'charitable' );

			$log_path = $this->get_log_file_path();

			// Only show link if file exists.
			if ( @file_exists( $log_path ) ) { // phpcs:ignore
				$log_url = $this->get_log_file_url();
				$help_text .= ' <a href="' . esc_url( $log_url ) . '" target="_blank">' . __( 'View log file', 'charitable' ) . '</a>';
			}

			return $help_text;
		}

		/**
		 * Check if test mode is enabled.
		 *
		 * @since  1.8.9
		 *
		 * @return bool
		 */
		protected function is_test_mode() {
			// Check if global test mode is enabled.
			if ( charitable_get_option( 'test_mode' ) ) {
				return true;
			}

			// Check if any gateway is in test mode.
			$gateways = Charitable_Gateways::get_instance();
			if ( ! $gateways ) {
				return false;
			}

			$active_gateways = $gateways->get_active_gateways();
			foreach ( $active_gateways as $gateway_id => $gateway ) {
				if ( method_exists( $gateway, 'is_test_mode' ) && $gateway->is_test_mode() ) {
					return true;
				}
				// Also check for test_mode option directly.
				if ( 'yes' === charitable_get_option( $gateway_id . '_test_mode', 'no' ) ) {
					return true;
				}
			}

			return false;
		}
	}

endif;

